<?php

namespace App\Filament\Blocks;

use Filament\Forms\Components\Builder\Block;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Toggle;

class PrismaHeroBlock
{
    public static function make(): Block
    {
        $isPremiumLayout = fn ($get): bool => ($get('layout_design') ?? 'premium') === 'premium';

        $hasTextContent = fn ($get): bool => (function () use ($get): bool {
            $layout = $get('layout_design') ?? 'premium';
            $premiumMode = $get('premium_display_mode') ?? 'full';
            $showHeroText = (bool) ($get('show_hero_text') ?? true);

            if ($layout === 'image_only') {
                return false;
            }

            if ($layout === 'premium' && $premiumMode === 'image_only') {
                return false;
            }

            return $showHeroText;
        })();

        $hasImageContent = fn ($get): bool => (function () use ($get): bool {
            $layout = $get('layout_design') ?? 'premium';
            $premiumMode = $get('premium_display_mode') ?? 'full';
            $showBackgroundImage = (bool) ($get('show_background_image') ?? true);

            if (! $showBackgroundImage) {
                return false;
            }

            if ($layout === 'text_only') {
                return false;
            }

            if ($layout === 'premium' && $premiumMode === 'text_only') {
                return false;
            }

            return true;
        })();

        $supportsImageConfiguration = fn ($get): bool => (function () use ($get): bool {
            $layout = $get('layout_design') ?? 'premium';
            $premiumMode = $get('premium_display_mode') ?? 'full';

            if ($layout === 'text_only') {
                return false;
            }

            if ($layout === 'premium' && $premiumMode === 'text_only') {
                return false;
            }

            return true;
        })();

        return Block::make('prisma_hero')
            ->label('بانر رئيسي (Prisma)')
            ->icon('heroicon-o-home')
            ->schema([
                Section::make('Layout Design')
                    ->schema([
                        Select::make('layout_design')
                            ->label('Hero Layout Style')
                            ->options([
                                'premium' => 'Premium (Floating Cards)',
                                'center_text_on_image' => 'Center Text on Image',
                                'text_beside_image' => 'Text Beside Image',
                                'text_below_image' => 'Text Below Image',
                                'image_only' => 'Image Only (No Text)',
                                'text_only' => 'Text Only (No Image)',
                            ])
                            ->default('premium')
                            ->live()
                            ->helperText('Choose how your hero section looks'),

                        Select::make('premium_display_mode')
                            ->label('Premium Display Mode')
                            ->options([
                                'full' => 'Image + Text + Categories',
                                'image_only' => 'Image Only',
                                'text_only' => 'Text Only',
                            ])
                            ->default('full')
                            ->live()
                            ->visible($isPremiumLayout),

                        Toggle::make('show_background_image')
                            ->label('Show Background Image')
                            ->default(true)
                            ->live(),

                        Toggle::make('show_hero_text')
                            ->label('Show Hero Text')
                            ->default(true)
                            ->live(),

                        Toggle::make('show_cta_button')
                            ->label('Show CTA Button')
                            ->default(true)
                            ->visible($hasTextContent),

                        Toggle::make('show_categories')
                            ->label('Show Category Cards')
                            ->default(true)
                            ->visible($isPremiumLayout),

                        Repeater::make('categories')
                            ->label('Category Cards')
                            ->schema([
                                Grid::make(2)
                                    ->schema([
                                        TextInput::make('name')
                                            ->label('Category Name')
                                            ->required()
                                            ->live(debounce: 700)
                                            ->placeholder('e.g., الفحص الشامل'),

                                        TextInput::make('url')
                                            ->label('Link URL')
                                            ->default('#')
                                            ->live(debounce: 700)
                                            ->placeholder('#packages'),
                                    ]),

                                Toggle::make('use_image')
                                    ->label('Use Image (instead of icon)')
                                    ->default(true)
                                    ->live(),

                                FileUpload::make('image')
                                    ->label('Category Image')
                                    ->image()
                                    ->disk('public')
                                    ->directory('category-images')
                                    ->visibility('public')
                                    ->imageEditor()
                                    ->imageEditorAspectRatios(['4:3', '16:9', '1:1'])
                                    ->maxSize(2048)
                                    ->visible(fn ($get) => $get('use_image') === true),

                                Select::make('icon')
                                    ->label('Icon (fallback)')
                                    ->options(\App\Support\IconOptions::fontAwesomeOptions())
                                    ->default('fas fa-vial')
                                    ->searchable()
                                    ->visible(fn ($get) => $get('use_image') !== true),
                            ])
                            ->columns(1)
                            ->maxItems(8)
                            ->defaultItems(0)
                            ->addActionLabel('Add Category')
                            ->visible(fn ($get) => $isPremiumLayout($get) && (bool) ($get('show_categories') ?? true))
                            ->collapsible()
                            ->itemLabel(fn (array $state): ?string => $state['name'] ?? null),

                        Toggle::make('full_width')
                            ->label('Full Width')
                            ->default(true),

                        Toggle::make('rounded_corners')
                            ->label('Rounded Corners')
                            ->default(false),

                        Toggle::make('add_shadow')
                            ->label('Add Shadow Effect')
                            ->default(false),
                    ])
                    ->columns(2),

                Section::make('المحتوى')
                    ->description('محتوى البانر (يختلف حسب اللغة المختارة)')
                    ->icon('heroicon-o-document-text')
                    ->schema([
                        TextInput::make('title')
                            ->label('العنوان الرئيسي')
                            ->placeholder('مثال: صحتك تبدأ بفحص دقيق')
                            ->live(debounce: 700)
                            ->visible($hasTextContent),
                        Textarea::make('subtitle')
                            ->label('العنوان الفرعي')
                            ->rows(2)
                            ->placeholder('وصف قصير...')
                            ->live(debounce: 700)
                            ->visible($hasTextContent),
                        TextInput::make('cta_text')
                            ->label('نص الزر')
                            ->placeholder('احجز الآن')
                            ->live(debounce: 700)
                            ->visible(fn ($get) => $hasTextContent($get) && (bool) ($get('show_cta_button') ?? true)),
                    ])
                    ->collapsible()
                    ->collapsed(false)
                    ->visible($hasTextContent),

                Section::make('Button & Links')
                    ->schema([
                        TextInput::make('cta_url')
                            ->label('CTA URL')
                            ->default('#packages')
                            ->live(debounce: 700)
                            ->visible(fn ($get) => $hasTextContent($get) && (bool) ($get('show_cta_button') ?? true)),

                        Select::make('button_style')
                            ->label('Button Style')
                            ->options([
                                'primary' => 'Primary (Gradient)',
                                'secondary' => 'Secondary (Outline)',
                                'white' => 'White',
                                'ghost' => 'Ghost',
                            ])
                            ->default('primary')
                            ->visible(fn ($get) => $hasTextContent($get) && (bool) ($get('show_cta_button') ?? true)),

                        ColorPicker::make('cta_bg_start')
                            ->label('CTA Gradient Start')
                            ->default('#0099FF')
                            ->visible(fn ($get) => $hasTextContent($get) && (bool) ($get('show_cta_button') ?? true)),

                        ColorPicker::make('cta_bg_end')
                            ->label('CTA Gradient End')
                            ->default('#7B00FF')
                            ->visible(fn ($get) => $hasTextContent($get) && (bool) ($get('show_cta_button') ?? true)),

                        ColorPicker::make('cta_text_color')
                            ->label('CTA Text Color')
                            ->default('#ffffff')
                            ->visible(fn ($get) => $hasTextContent($get) && (bool) ($get('show_cta_button') ?? true)),
                    ])
                    ->visible($hasTextContent),

                Section::make('Image')
                    ->schema([
                        FileUpload::make('background_image')
                            ->label('Hero Image')
                            ->image()
                            ->disk('public')
                            ->directory('hero')
                            ->visibility('public')
                            ->helperText('Recommended: 1920x600 for full width, 800x600 for side layouts')
                            ->visible($supportsImageConfiguration),

                        Toggle::make('image_has_text')
                            ->label('Image already has text')
                            ->default(false)
                            ->helperText('Enable if your image already contains the text (no text layer will be shown)')
                            ->visible(fn ($get) => $hasImageContent($get) && $get('layout_design') === 'center_text_on_image'),

                        Select::make('image_position')
                            ->label('Image Position')
                            ->options([
                                'left' => 'Left',
                                'right' => 'Right',
                            ])
                            ->default('right')
                            ->visible(fn ($get) => $hasImageContent($get) && $get('layout_design') === 'text_beside_image'),

                        Select::make('image_height')
                            ->label('Image Height')
                            ->options([
                                'small' => 'Small (300px)',
                                'medium' => 'Medium (450px)',
                                'large' => 'Large (600px)',
                                'fullscreen' => 'Full Screen',
                            ])
                            ->default('medium')
                            ->visible($supportsImageConfiguration),
                    ]),

                Section::make('Background & Colors')
                    ->schema([
                        ColorPicker::make('bg_color')
                            ->label('Background Color')
                            ->default('#0a0a0a'),

                        ColorPicker::make('overlay_color')
                            ->label('Image Overlay Color')
                            ->default('#000000')
                            ->visible($hasImageContent),

                        TextInput::make('overlay_opacity')
                            ->label('Overlay Opacity (0-100)')
                            ->numeric()
                            ->default(35)
                            ->suffix('%')
                            ->live(debounce: 700)
                            ->visible($hasImageContent),

                        ColorPicker::make('text_color')
                            ->label('Text Color')
                            ->default('#ffffff')
                            ->visible($hasTextContent),

                        Toggle::make('show_pattern')
                            ->label('Show Background Pattern')
                            ->default(true)
                            ->visible(fn ($get) => $get('layout_design') === 'center_text_on_image'),
                    ])
                    ->collapsed(),
            ]);
    }
}
