<?php

namespace App\Filament\Resources;

use App\Filament\Resources\NavigationResource\Pages;
use App\Models\Navigation;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class NavigationResource extends Resource
{
    protected static ?string $model = Navigation::class;
    protected static ?string $navigationIcon = 'heroicon-o-bars-3';
    protected static ?string $navigationLabel = 'Menu Manager';
    protected static ?string $navigationGroup = 'Content';
    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Menu Item Details')
                    ->schema([
                        Forms\Components\TextInput::make('title')
                            ->label('الاسم بالعربي')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('مثال: الرئيسية')
                            ->helperText('يُعرض عند اختيار اللغة العربية'),

                        Forms\Components\TextInput::make('title_translations.en')
                            ->label('الاسم بالإنجليزي')
                            ->maxLength(255)
                            ->placeholder('e.g. Home')
                            ->helperText('يُعرض عند اختيار اللغة الإنجليزية'),
                        
                        Forms\Components\Select::make('location')
                            ->label('Menu Location')
                            ->options([
                                'header' => 'Header Menu',
                                'footer' => 'Footer Menu',
                            ])
                            ->default('header')
                            ->required(),
                        
                        Forms\Components\Select::make('type')
                            ->label('Link Type')
                            ->options([
                                'custom' => 'Custom URL',
                                'page' => 'Page',
                                'package' => 'Package',
                            ])
                            ->default('custom')
                            ->live()
                            ->required(),
                        
                        Forms\Components\TextInput::make('url')
                            ->label('Custom URL')
                            ->placeholder('e.g., /about or https://example.com')
                            ->maxLength(255)
                            ->visible(fn ($get) => $get('type') === 'custom'),
                        
                        Forms\Components\Select::make('page_id')
                            ->label('Select Page')
                            ->relationship('page', 'title', fn ($query) => $query->where('is_published', true))
                            ->searchable()
                            ->preload()
                            ->visible(fn ($get) => $get('type') === 'page')
                            ->required(fn ($get) => $get('type') === 'page'),
                        
                        Forms\Components\Select::make('package_id')
                            ->label('Select Package')
                            ->relationship('package', 'name', fn ($query) => $query->where('is_active', true))
                            ->searchable()
                            ->preload()
                            ->visible(fn ($get) => $get('type') === 'package')
                            ->required(fn ($get) => $get('type') === 'package'),
                        
                        Forms\Components\Select::make('parent_id')
                            ->label('Parent Menu Item')
                            ->relationship('parent', 'title', fn ($query) => $query->whereNull('parent_id'))
                            ->placeholder('None (Top Level)')
                            ->helperText('Select a parent to create a dropdown menu item'),
                        
                        Forms\Components\Select::make('target')
                            ->label('Open In')
                            ->options([
                                '_self' => 'Same Tab',
                                '_blank' => 'New Tab',
                            ])
                            ->default('_self')
                            ->required(),
                        
                        Forms\Components\TextInput::make('sort_order')
                            ->label('Sort Order')
                            ->numeric()
                            ->default(0)
                            ->helperText('Lower numbers appear first'),
                        
                        Forms\Components\Toggle::make('is_active')
                            ->label('Active')
                            ->default(true),
                    ])
                    ->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->sortable()
                    ->weight('font-bold'),
                
                Tables\Columns\TextColumn::make('location')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'header' => 'success',
                        'footer' => 'warning',
                    }),
                
                Tables\Columns\TextColumn::make('type')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'custom' => 'gray',
                        'page' => 'info',
                        'package' => 'primary',
                    }),
                
                Tables\Columns\TextColumn::make('parent.title')
                    ->label('Parent')
                    ->placeholder('Top Level'),
                
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),
                
                Tables\Columns\TextColumn::make('sort_order')
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('location')
                    ->options([
                        'header' => 'Header Menu',
                        'footer' => 'Footer Menu',
                    ]),
                
                Tables\Filters\SelectFilter::make('type')
                    ->options([
                        'custom' => 'Custom URL',
                        'page' => 'Page',
                        'package' => 'Package',
                    ]),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('sort_order', 'asc')
            ->reorderable('sort_order');
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->with('parent')
            ->orderBy('sort_order');
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => \App\Filament\Resources\NavigationResource\Pages\ListNavigations::route('/'),
            'create' => \App\Filament\Resources\NavigationResource\Pages\CreateNavigation::route('/create'),
            'edit' => \App\Filament\Resources\NavigationResource\Pages\EditNavigation::route('/{record}/edit'),
        ];
    }
}
