<?php

namespace Illuminate\Support;

/**
 * Number polyfill for systems without intl extension
 * This overrides Laravel's Number class
 */
class Number
{

    protected static $locale = 'en';
    protected static $currency = 'USD';

    public static function format(int|float $number, ?int $precision = null, ?int $maxPrecision = null, ?string $locale = null)
    {
        if (! is_null($maxPrecision)) {
            return number_format($number, $maxPrecision);
        } elseif (! is_null($precision)) {
            return number_format($number, $precision);
        }
        return number_format($number, 0);
    }

    public static function parse(string $string, ?int $type = null, ?string $locale = null): int|float|false
    {
        return (float) $string;
    }

    public static function parseInt(string $string, ?string $locale = null): int|false
    {
        return (int) $string;
    }

    public static function parseFloat(string $string, ?string $locale = null): float|false
    {
        return (float) $string;
    }

    public static function spell(int|float $number, ?string $locale = null, ?int $after = null, ?int $until = null)
    {
        return (string) $number;
    }

    public static function ordinal(int|float $number, ?string $locale = null)
    {
        return $number . 'th';
    }

    public static function spellOrdinal(int|float $number, ?string $locale = null)
    {
        return $number . 'th';
    }

    public static function percentage(int|float $number, int $precision = 0, ?int $maxPrecision = null, ?string $locale = null)
    {
        return number_format($number, $precision) . '%';
    }

    public static function currency(int|float $number, string $in = '', ?string $locale = null, ?int $precision = null)
    {
        $currency = ! empty($in) ? $in : static::$currency;
        $decimals = $precision ?? 2;
        return $currency . ' ' . number_format($number, $decimals);
    }

    public static function fileSize(int|float $bytes, int $precision = 0, ?int $maxPrecision = null)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
        $unitCount = count($units);

        for ($i = 0; ($bytes / 1024) > 0.9 && ($i < $unitCount - 1); $i++) {
            $bytes /= 1024;
        }

        return sprintf('%s %s', static::format($bytes, $precision, $maxPrecision), $units[$i]);
    }

    public static function abbreviate(int|float $number, int $precision = 0, ?int $maxPrecision = null)
    {
        return static::forHumans($number, $precision, $maxPrecision, abbreviate: true);
    }

    public static function forHumans(int|float $number, int $precision = 0, ?int $maxPrecision = null, bool $abbreviate = false)
    {
        return static::summarize($number, $precision, $maxPrecision, $abbreviate ? [
            3 => 'K',
            6 => 'M',
            9 => 'B',
            12 => 'T',
            15 => 'Q',
        ] : [
            3 => ' thousand',
            6 => ' million',
            9 => ' billion',
            12 => ' trillion',
            15 => ' quadrillion',
        ]);
    }

    public static function summarize(int|float $number, int $precision = 0, ?int $maxPrecision = null, array $units = [])
    {
        if (empty($units)) {
            $units = [
                3 => 'K',
                6 => 'M',
                9 => 'B',
                12 => 'T',
                15 => 'Q',
            ];
        }

        if ($number === 0) {
            return $precision > 0 ? static::format($number, $precision, $maxPrecision) : '0';
        }

        if ($number < 0) {
            return sprintf('-%s', static::summarize(abs($number), $precision, $maxPrecision, $units));
        }

        if ($number >= 1e15) {
            return sprintf('%s'.end($units), static::summarize($number / 1e15, $precision, $maxPrecision, $units));
        }

        $numberExponent = floor(log10($number));
        $displayExponent = $numberExponent - ($numberExponent % 3);
        $number /= pow(10, $displayExponent);

        return trim(sprintf('%s%s', static::format($number, $precision, $maxPrecision), $units[$displayExponent] ?? ''));
    }

    public static function clamp(int|float $number, int|float $min, int|float $max)
    {
        return min(max($number, $min), $max);
    }

    public static function pairs(int|float $to, int|float $by, int|float $start = 0, int|float $offset = 1)
    {
        $output = [];

        for ($lower = $start; $lower < $to; $lower += $by) {
            $upper = $lower + $by - $offset;

            if ($upper > $to) {
                $upper = $to;
            }

            $output[] = [$lower, $upper];
        }

        return $output;
    }

    public static function trim(int|float $number)
    {
        return json_decode(rtrim(rtrim(json_encode($number), '0'), '.'));
    }

    public static function useLocale(string $locale)
    {
        static::$locale = $locale;
    }

    public static function useCurrency(string $currency)
    {
        static::$currency = $currency;
    }
}
